const ApiService = () => {

    const _sendRequest = async (url, headers, method, data, onSuccessHandler, onErrorHandler) => {
        try {
            let options = {
                method: method,
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                    ...headers
                }
            };
            if (method === CONSTANTS.REQUEST_METHODS.POST) {
                options.body = data;
            }
            const response = await fetch(url, options);
            const textResponse = await response.text();

            onSuccessHandler(_isJsonString(textResponse) ? JSON.parse(textResponse) : textResponse, response.status);
        } catch (error) {
            let errorMessage = CONSTANTS.NOTIFICATION_MESSAGES.UNEXPECTED_ERROR;
            if (error.message.includes(CONSTANTS.NOTIFICATION_MESSAGES.FAILED_TO_FETCH)) {
                errorMessage = errorMessage.concat('. ', error.message);
            }
            onErrorHandler(errorMessage);
        }
    };

    const _isJsonString = str => {
        try {
            return JSON.parse(str) && !!str;
        } catch (e) {
            return false;
        }
    };

    const _sendGetRequest = async (url, headers, onSuccessHandler, onErrorHandler) => {
        await _sendRequest(url, headers, CONSTANTS.REQUEST_METHODS.GET, null, onSuccessHandler, onErrorHandler);
    };

    const _sendPostRequest = async (url, headers, data, onSuccessHandler, onErrorHandler) => {
        await _sendRequest(url, headers, CONSTANTS.REQUEST_METHODS.POST, data, onSuccessHandler, onErrorHandler);
    };

    const _onSuccessHandler = (data, status, message, isBasicAuth, additionalMessage = null, callback = null,
                               withFadeOut = true, innerToastNotification = false) => {
        let resultMessage;
        let type;

        switch (status) {
            case 200:
            case 201:
            case 204:
                resultMessage = message;
                if (additionalMessage) {
                    resultMessage = additionalMessage.concat('. ', resultMessage);
                }
                type = CONSTANTS.NOTIFICATION_TYPES.SUCCESS;
                break;
            case 400:
                resultMessage = CONSTANTS.NOTIFICATION_MESSAGES.ERROR_EXECUTING_REQUEST.replace('$ERROR', data.message);
                type = CONSTANTS.NOTIFICATION_TYPES.ERROR;
                break;
            case 401:
                resultMessage = isBasicAuth ?
                    CONSTANTS.NOTIFICATION_MESSAGES.WRONG_CREDENTIALS :
                    CONSTANTS.NOTIFICATION_MESSAGES.WRONG_CREDENTIALS_API_KEY;
                resultMessage = CONSTANTS.NOTIFICATION_MESSAGES.ERROR_EXECUTING_REQUEST.replace('$ERROR', resultMessage);
                type = CONSTANTS.NOTIFICATION_TYPES.ERROR;
                break;
            default:
                resultMessage = CONSTANTS.NOTIFICATION_MESSAGES.UNEXPECTED_RESPONSE
                    .replace('$STATUS', status).replace('$DATA', JSON.stringify(data));
                resultMessage = CONSTANTS.NOTIFICATION_MESSAGES.ERROR_EXECUTING_REQUEST.replace('$ERROR', resultMessage);
                type = CONSTANTS.NOTIFICATION_TYPES.ERROR;
                break;
        }

        setTimeout(() => {
            if (withFadeOut) {
                $("#overlay").fadeOut(300);
            }

            if (innerToastNotification) {
                if (type === CONSTANTS.NOTIFICATION_TYPES.SUCCESS) {
                    _showToast(resultMessage, CONSTANTS.NOTIFICATION_TYPES.SUCCESS);
                } else {
                    if (additionalMessage) {
                        _showToast(additionalMessage, CONSTANTS.NOTIFICATION_TYPES.SUCCESS);
                    }
                    _showToast(resultMessage, CONSTANTS.NOTIFICATION_TYPES.ERROR);
                }
            } else {
                if (type === CONSTANTS.NOTIFICATION_TYPES.SUCCESS) {
                    _sendMessage(resultMessage, type);
                } else {
                    if (additionalMessage) {
                        _sendMessage(additionalMessage, CONSTANTS.NOTIFICATION_TYPES.SUCCESS);
                    }
                    _sendMessage(resultMessage, type);
                }
            }

            if (callback) {
                callback();
            }
        }, 500);
    };

    const _onErrorHandler = (errorMessage, callback = null, withFadeOut = true,
                             innerToastNotification = false) => {
        let resultMessage = CONSTANTS.NOTIFICATION_MESSAGES.SOMETHING_WENT_WRONG;

        setTimeout(() => {
            if (withFadeOut) {
                $("#overlay").fadeOut(300);
            }

            if (innerToastNotification) {
                _showToast(resultMessage, CONSTANTS.NOTIFICATION_TYPES.ERROR);
            } else {
                _sendMessage(resultMessage, CONSTANTS.NOTIFICATION_TYPES.ERROR);
            }

            if (callback) {
                callback();
            }
        }, 500);
    };

    const _normalizeURL = url => {
        if (!url.startsWith('https://') && !url.startsWith('http://')) {
            url = "https://" + url;
        }
        try {
            return new URL(url).origin;
        } catch (error) {
            console.debug(`${new Date(Date.now()).toISOString()} -> [API_SERVICE] Invalid url provided: ${error}`);
            return url;
        }
    };

    const _showToast = (message, type) => {
        switch (type) {
            case CONSTANTS.NOTIFICATION_TYPES.SUCCESS:
                $('#successToastContent')[0].innerText = message;
                $('#successToastBody').toast('show');
                break;
            case CONSTANTS.NOTIFICATION_TYPES.ERROR:
                $('#errorToastContent')[0].innerText = message;
                $('#errorToastBody').toast('show');
                break;
            default:
                break;
        }
    };

    const _sendMessage = (message, type) => {
        chrome.runtime.sendMessage({
            action: CONSTANTS.ACTIONS.SEND_NOTIFICATION_MESSAGE,
            message,
            type
        });
    };

    ApiService.sendGetRequest = _sendGetRequest;
    ApiService.sendPostRequest = _sendPostRequest;
    ApiService.onSuccessHandler = _onSuccessHandler;
    ApiService.onErrorHandler = _onErrorHandler;
    ApiService.normalizeURL = _normalizeURL;
};